""" dilated_resnet.py
    Dilated convolutions for larger receptive field
    January 2021
"""

import torch
import torch.nn as nn
import torch.nn.functional as F

# Ignore statemenst for pylint:
#     Too many branches (R0912), Too many statements (R0915), No member (E1101),
#     Not callable (E1102), Invalid name (C0103), No exception (W0702)
# pylint: disable=R0912, R0915, E1101, E1102, C0103, W0702, R0914


class BasicBlock(nn.Module):
    """Basic residual block class"""
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(BasicBlock, self).__init__()
        self.conv1 = nn.Conv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=2, dilation=2, bias=False)

        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=1, padding=2, dilation=2, bias=False)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False)
            )

    def forward(self, x):
        out = F.relu(self.conv1(x))
        out = self.conv2(out)
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class ResNet(nn.Module):
    """Modified ResNet model class"""

    def __init__(self, block, num_blocks, num_classes=10, iters=5, feature_scale=1,
                 dataset="CIFAR10"):
        super(ResNet, self).__init__()
        self.dataset = dataset
        self.iters = iters
        self.in_planes = int(feature_scale*64)
        self.num_classes = num_classes
        self.conv1 = nn.Conv2d(3, int(feature_scale * 64), kernel_size=3,
                               stride=2, padding=2, dilation=2, bias=False)
        self.conv2 = nn.Conv2d(int(feature_scale * 64), int(feature_scale * 64),
                               kernel_size=3, stride=1, padding=2, dilation=2, bias=False)
        layers = []
        for j in range(self.iters):
            for i in range(len(num_blocks)):
                layers.append(self._make_layer(block, int(feature_scale*64), num_blocks[i], stride=1))

        self.recur_block = nn.Sequential(*layers)
        self.conv3 = nn.Conv2d(int(feature_scale*64), 512, kernel_size=3,
                               stride=2, padding=2, dilation=2, bias=False)
        self.linear = nn.Linear(512, num_classes)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1]*(num_blocks-1)
        layers = []
        for strd in strides:
            layers.append(block(self.in_planes, planes, strd))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x):

        out = F.relu(self.conv1(x))
        out = self.recur_block(out)
        thought = F.relu(self.conv3(out))

        if self.dataset.upper() == "CIFAR10":
            thought = F.avg_pool2d(thought, 8)
        elif self.dataset.upper() == "TINYIMAGENET":
            thought = F.avg_pool2d(thought, 16)

        thought = thought.view(thought.size(0), -1)
        thought = self.linear(thought)
        return thought


def resnet_7(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=1, feature_scale=8)


def resnet_11(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=2, feature_scale=8)


def resnet_15(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=3, feature_scale=8)


def resnet_19(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=4, feature_scale=8)


def resnet_23(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=5, feature_scale=8)


def resnet_27(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=6, feature_scale=8)


def resnet_7_tinyimagenet(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=1, feature_scale=8,
                  dataset="TINYIMAGENET")


def resnet_11_tinyimagenet(num_outputs=200):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=2, feature_scale=8,
                  dataset="TINYIMAGENET")


def resnet_15_tinyimagenet(num_outputs=200):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=3, feature_scale=8,
                  dataset="TINYIMAGENET")


def resnet_19_tinyimagenet(num_outputs=200):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=4, feature_scale=8,
                  dataset="TINYIMAGENET")


def resnet_23_tinyimagenet(num_outputs=200):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=5, feature_scale=8,
                  dataset="TINYIMAGENET")


def resnet_27_tinyimagenet(num_outputs=200):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=6, feature_scale=8,
                  dataset="TINYIMAGENET")

# -------------------------- 4.1 M --------------------------------------------------------
def resnet_width_depth_2_1_4_cifar10_4(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=1, feature_scale=4)

def resnet_width_depth_1_1_5_cifar10_4(num_outputs=10):
    return ResNet(BasicBlock, [1], num_classes=num_outputs, iters=1, feature_scale=5)

def resnet_width_depth_2_2_3_cifar10_4(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=2, feature_scale=3.1)

# ------------------------------------------------------------------------------------------

# ------------------------- 6. 5 M ---------------------------------------------------------
def resnet_width_depth_2_2_4_cifar10_6(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=2, feature_scale=4)

def resnet_width_depth_2_1_5_cifar10_6(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=1, feature_scale=5)

def resnet_width_depth_2_4_3_cifar10_6(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=4, feature_scale=3)

# ------------------------------------------------------------------------------------------

# ------------------------- 8.8 M ----------------------------------------------------------

def resnet_width_depth_2_3_4_cifar10_9(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=3, feature_scale=4)

def resnet_width_depth_2_1_6_cifar10_9(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=1, feature_scale=6)

def resnet_width_depth_2_2_5_cifar10_9(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=2, feature_scale=4.7)

# ------------------------------------------------------------------------------------------

# ------------------------- 11.1 M ---------------------------------------------------------
def resnet_width_depth_2_4_4_cifar10_11(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=4, feature_scale=4)

def resnet_width_depth_2_1_7_cifar10_11(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=1, feature_scale=7)

def resnet_width_depth_2_3_4_cifar10_11(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=3, feature_scale=4.5)

# ------------------------------------------------------------------------------------------

# ------------------------- 13.5 M ---------------------------------------------------------
def resnet_width_depth_2_5_4_cifar10_13(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=5, feature_scale=4)

def resnet_width_depth_2_1_8_cifar10_13(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=1, feature_scale=7.8)

def resnet_width_depth_2_3_5_cifar10_13(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=3, feature_scale=5)
# ------------------------------------------------------------------------------------------

# ------------------------- 15.9 M ---------------------------------------------------------
def resnet_width_depth_2_6_4_cifar10_16(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=6, feature_scale=4)

def resnet_width_depth_2_1_8_cifar10_16(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=1, feature_scale=8.5)

def resnet_width_depth_2_4_5_cifar10_16(num_outputs=10):
    return ResNet(BasicBlock, [2], num_classes=num_outputs, iters=4, feature_scale=4.8)
# ------------------------------------------------------------------------------------------


